<?php

/**
 * @file plugins/themes/default/AcademicProThemePlugin.inc.php
 *
 * Copyright (c) 2010-2020 openjournaltheme.com
 * Copyright (c) 2010-2020 openjournaltheme team
 * Read this term of use of this theme here : https://openjournaltheme.com/term-of-conditions/
 *
 * Modify, redistribute or make commercial copy of this part or whole of this code is prohibited without written permission from openjournaltheme.com
 * Modified by openjournaltheme.com
 * contact : openjournaltheme@gmail.com
 *
 * @class AcademicProThemePlugin
 * @ingroup plugins_themes_bootstrap3
 *
 * @brief Default theme
 */

//make compatible with ojs 3.2.0-1
if (!class_exists('\Academic\OpenJournalTheme')) {
    require_once __DIR__ . '/include/Core.php';
}

use Academic\OpenJournalTheme;

import('lib.pkp.classes.plugins.ThemePlugin');

//make compatible with ojs 3.2.0-1
OpenJournalTheme::importClass('classes.core.ServicesContainer'); //used on ojs 3.1.2
OpenJournalTheme::importClass('classes.core.Services'); //used on ojs 3.2.0-1

class AcademicProThemePlugin extends ThemePlugin
{
    public $pluginDir;

    public function __construct()
    {
        $this->pluginDir = $this->getPluginPath();
    }

    /**
     * Get the display name of this plugin
     * @return string
     */
    public function getDisplayName()
    {
        return 'Academic Pro - For OJS 3.2.X';
    }

    public $statisticOption;


    public function check_now()
    {
        $post = (object) $_POST;
        $openJournalTheme = $this->openJournalThemeInit();
        if (!$openJournalTheme->isFromLocal()) {
            $openJournalTheme->checkNow($post->hash);
            return $openJournalTheme->getStatusValidated();
        }
        return 1;
    }

    public function getPluginName()
    {
        return strtolower(__CLASS__);
    }

    public function validate_now()
    {
        $license = $_POST['license'];
        $key = $_POST['hash'];
        $openJournalTheme = $this->openJournalThemeInit();
        if ($openJournalTheme->authHash($key) && $openJournalTheme->isFromLocal()) {
            $openJournalTheme->updateSetting('license', $license);
            $openJournalTheme->saveOption('licenseMain', $license);
            $openJournalTheme->validate();
            return $openJournalTheme->getStatusValidated();
        }
        return 1;
    }

    /**
     * @return object OpenJournalValidation object
     */
    public function openJournalThemeInit()
    {
        return new OpenJournalTheme($this, false);
    }

    public function setPageHandler($hookName, $args)
    {
        $page = $args[0];
        if ($page == $this->getPluginName()) {
            $this->import('AcademicProHandler');
            define('HANDLER_CLASS', 'AcademicProHandler');
            return true;
        }
        return false;
    }

    function register($category, $path, $mainContextId = null)
    {
        if (!parent::register($category, $path, $mainContextId)) return false;

        // Don't perform any futher operations if theme is not currently active
        if (!$this->isActive()) {
            return true;
        }


        //view credit to openjournaltheme.com
        HookRegistry::register('LoadHandler', [$this, 'setPageHandler']);
        HookRegistry::register('TemplateManager::display', [$this, 'addHeaderMeta']);

        return true;
    }

    public function addHeaderMeta($hookName, $args)
    {
        $templateMgr            = &$args[0];
        $templateMgr->addHeader(
            $this->getName(),
            '<meta name="' . $this->getName() . '" content="' . $this->getDisplayName() . ' Version ' . $this->getPluginVersion() . ' by openjournaltheme.com">',
            [
                'contexts' => ['frontend'],
            ]
        );
    }

    /**
     * Initialize the theme
     *
     * @return null
     */
    public function init()
    {
        AppLocale::requireComponents(LOCALE_COMPONENT_PKP_MANAGER, LOCALE_COMPONENT_APP_MANAGER);

        $application = Application::get();


        // license
        $this->addOption('licenseMain', 'FieldText', [
            'label'       => 'Product License Code',
            'description' => 'Enter License key that you get on the email when you have complete the order of this product',
            'default'     => null,
        ]);

        //header option
        $this->addOption('headerTheme', 'FieldOptions', [
            'type'        => 'radio',
            'label'       => 'Header design ',
            'description' => 'Select Prebuilt design for header',
            'options'     => [
                [
                    'value' => 'red',
                    'label' => __('plugins.themes.academic_pro.headerTheme.red'),
                ],
                [
                    'value' => 'edu',
                    'label' => 'Flatter Focused',
                ],
                [
                    'value' => 'dots',
                    'label' => __('plugins.themes.academic_pro.headerTheme.dots'),
                ],
                [
                    'value' => 'diamond',
                    'label' => __('plugins.themes.academic_pro.headerTheme.diamond'),
                ],
                [
                    'value' => 'blue',
                    'label' => __('plugins.themes.academic_pro.headerTheme.blue'),
                ],
            ],
            'default'     => 'red',
        ]);

        //logo feature
        // Show or hide the statistic
        $this->addOption('logoStyleBig', 'FieldOptions', [
            'type'          => 'radio',
            'label'         => __('plugins.themes.academic_pro.logo_style'),
            'description'   => __('plugins.themes.academic_pro.logo_style_description'),
            'options'       => [
                [
                    'value' => true,
                    'label' => __('plugins.themes.academic_pro.logo_big'),
                ],
                [
                    'value' => false,
                    'label' => __('plugins.themes.academic_pro.logo_small'),
                ],
            ],
            'default'     => false,
        ]);

        // Add theme options
        $this->addOption('baseColour', 'FieldColor', [
            'label'       => __('plugins.themes.academic_pro.theme_color'),
            'description' => __('plugins.themes.academic_pro.theme_color_desc'),
            'default'     => '#8a0493',
        ]);

        // Show or hide the statistic
        $this->addOption('statistics', 'FieldOptions', [
            'label'       => __('plugins.themes.academic_pro.stat_label'),
            'description' => __('plugins.themes.academic_pro.stat.desc'),
            'options'     => [
                [
                    'value' => true,
                    'label' => __('plugins.themes.academic_pro.enable'),
                ],
            ],
            'default'     => true,
        ]);

        // Show or hide the article tracking date
        $this->addOption('article_tracking_date', 'FieldOptions', [
            'label'       => __('plugins.themes.academic_pro.article_tracking_date'),
            'description' => __('plugins.themes.academic_pro.article_tracking_date.desc'),
            'options'     => [
                [
                    'value' => true,
                    'label' => __('plugins.themes.academic_pro.enable'),
                ],
            ],
            'default'     => true,
        ]);

        // Show or hide the galley link on journal index
        $this->addOption('galley_link_on_index', 'FieldOptions', [
            'label'       => __('plugins.themes.academic_pro.galley_link_on_index'),
            'description' => __('plugins.themes.academic_pro.galley_link_on_index_desc'),
            'options'     => [
                [
                    'value' => true,
                    'label' => __('plugins.themes.academic_pro.enable'),
                ],
            ],
            'default'     => true,
        ]);

        // Show or hide dimension badge
        $this->addOption('dimension_badge', 'FieldOptions', [
            'label'       => __('plugins.themes.academic_pro.dimension_badge_label'),
            'description' => __('plugins.themes.academic_pro.dimension_badge_desc'),
            'options'     => [
                [
                    'value' => true,
                    'label' => __('plugins.themes.academic_pro.enable'),
                ],
            ],
            'default'     => false,
        ]);

        // Show or hide the galley link on journal index
        $this->addOption('doi_on_index', 'FieldOptions', [
            'label'       => __('plugins.themes.academic_pro.doi_on_index'),
            'description' => __('plugins.themes.academic_pro.doi_on_index_desc'),
            'options'     => [
                [
                    'value' => true,
                    'label' => __('plugins.themes.academic_pro.enable'),
                ],
            ],
            'default'     => true,
        ]);

        // Enable crossmark on article page
        $this->addOption('crossmark_button', 'FieldOptions', [
            'label'       => __('plugins.themes.academic_pro.crossmark_label'),
            'description' => __('plugins.themes.academic_pro.crossmark_button_desc'),
            'options'     => [
                [
                    'value' => true,
                    'label' => __('plugins.themes.academic_pro.enable'),
                ],
            ],
            'default'     => false,
        ]);

        $this->addOption('maintenance_mode', 'FieldOptions', [
            'label'       => __('plugins.themes.academic_pro.maintenance_label'),
            'description' => __('plugins.themes.academic_pro.maintenance_desc'),
            'options'     => [
                [
                    'value' => true,
                    'label' => __('plugins.themes.academic_pro.enable'),
                ],
            ],
            'default'     => false,
        ]);

        $this->addOption('author_affiliation', 'FieldOptions', [
            'label'       => __('plugins.themes.academic_pro.author_affiliation_label'),
            'description' => __('plugins.themes.academic_pro.author_affiliation_desc'),
            'options'     => [
                [
                    'value' => true,
                    'label' => __('plugins.themes.academic_pro.enable'),
                ],
            ],
            'default'     => false,
        ]);

        //---- end option

        $headerTheme = $this->getOption('headerTheme');
        if (empty($headerTheme)) {
            $this->addStyle('headerTheme', 'styles/header/red.css');
        } else {
            $this->addStyle('headerTheme-' . $headerTheme, 'styles/header/' . $headerTheme . '.css');
        }

        // Update colour based on theme option
        $additionalLessVariables = [];

        $additionalLessVariables[] = '@theme-color:' . $this->getOption('baseColour') . ';';

        if ($this->isColourDark($this->getOption('baseColour'))) {
            $additionalLessVariables[] = '@font-color: desaturate(lighten(@theme-color, 81%), 15%);';
            $additionalLessVariables[] = '@middle-color: lighten(@theme-color, 41%);';
        }
        if (!$this->isColourDark($this->getOption('baseColour'))) {
            $additionalLessVariables[] = '@font-color: darken(@theme-color, 75%);';
            $additionalLessVariables[] = '@middle-color: darken(@theme-color, 1%);';
        }

        // Register option for bootstrap themes

        $this->addStyle('bootstrap', 'styles/paper.less');

        $locale = AppLocale::getLocale();
        if (AppLocale::getLocaleDirection($locale) === 'rtl') {
            if (Config::getVar('general', 'enable_cdn')) {
                $this->addStyle('bootstrap-rtl', '//cdn.rawgit.com/morteza/bootstrap-rtl/v3.3.4/dist/css/bootstrap-rtl.min.css', ['baseUrl' => '']);
            } else {
                $this->addStyle('bootstrap-rtl', 'styles/bootstrap-rtl.min.css');
            }
        }

        // Load jQuery from a CDN or, if CDNs are disabled, from a local copy.
        $min     = Config::getVar('general', 'enable_minified') ? '.min' : '';

        $request = $application->getRequest();


        // Use OJS's built-in jQuery files
        $jquery   = $request->getBaseUrl() . '/lib/pkp/lib/vendor/components/jquery/jquery' . $min . '.js';
        $jqueryUI = $request->getBaseUrl() . '/lib/pkp/lib/vendor/components/jqueryui/jquery-ui' . $min . '.js';


        // Use an empty `baseUrl` argument to prevent the theme from looking for
        // the files within the theme directory
        $this->addScript('jQuery', $jquery, ['baseUrl' => '']);
        $this->addScript('jQueryUI', $jqueryUI, ['baseUrl' => '']);
        $this->addScript('jQueryTagIt', $request->getBaseUrl() . '/lib/pkp/js/lib/jquery/plugins/jquery.tag-it.js', ['baseUrl' => '']);

        // Load Bootstrap
        $this->addScript('bootstrap', 'bootstrap/js/bootstrap.min.js');


        //load Academic pro script
        $this->addScript('sticky', 'js/lib/sticky.min.js');
        $this->addScript('readmore', 'js/lib/readmore.min.js');
        $this->addScript('jats_toc', 'js/lib/jats.toc.min.js');

        //load readmore library
        $this->addScript('academic-pro', 'js/app.js');

        //slider
        $this->addStyle('pxslider-css', 'https://cdnjs.cloudflare.com/ajax/libs/slick-carousel/1.9.0/slick.min.css', ['baseUrl' => '']);
        $this->addStyle('pxslider-css-theme', 'https://cdnjs.cloudflare.com/ajax/libs/slick-carousel/1.9.0/slick-theme.min.css', ['baseUrl' => '']);
        $this->addScript('pxslider', 'https://cdnjs.cloudflare.com/ajax/libs/slick-carousel/1.9.0/slick.min.js', ['baseUrl' => '']);

        // Add navigation menu areas for this theme
        $this->addMenuArea(['primary', 'user']);

        $this->addStyle('google-font', 'https://fonts.googleapis.com/css?family=Lora', ['baseUrl' => '']);

        $this->addStyle('fontawesome', 'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/4.7.0/css/font-awesome.css', ['baseUrl' => '']);

        // Load theme stylesheet and script
        $this->addStyle('stylesheet', 'styles/color-preferences.less');
        $this->modifyStyle('stylesheet', ['addLessVariables' => join($additionalLessVariables)]);

        // enable minify if not in development
        $isCoreCssMinify = $this->getOption('maintenance_mode');

        // vd($this->getOption('maintenance_mode'));

        if ($isCoreCssMinify) {
            $this->addStyle('academic_pro', 'styles/academic_pro.css');
        } else {
            $this->addStyle('academic_pro', 'styles/academic_pro.min.css');
        }

        $this->addStyle('mobile_responsive', 'styles/mobile_responsive.css');

        HookRegistry::register('TemplateManager::display', [$this, 'loadTemplateData']);
    }

    /**
     * Get the description of this plugin
     * @return string
     */
    public function getDescription()
    {
        return __('plugins.themes.academic_pro.description');
    }

    public function real_strip_tags($string, &$smarty)
    {
        return trim(preg_replace('/ +/', ' ', preg_replace('/[^A-Za-z0-9 ]/', ' ', urldecode(html_entity_decode(strip_tags($string))))));
    }

    public function hex2rgba($color, $opacity = false)
    {
        $default = 'rgb(0,0,0)';

        //Return default if no color provided
        if (empty($color)) {
            return $default;
        }

        //Sanitize $color if "#" is provided
        if ($color[0] == '#') {
            $color = substr($color, 1);
        }

        //Check if color has 6 or 3 characters and get values
        if (strlen($color) == 6) {
            $hex = [$color[0] . $color[1], $color[2] . $color[3], $color[4] . $color[5]];
        } elseif (strlen($color) == 3) {
            $hex = [$color[0] . $color[0], $color[1] . $color[1], $color[2] . $color[2]];
        } else {
            return $default;
        }

        //Convert hexadec to rgb
        $rgb = array_map('hexdec', $hex);

        //Check if opacity is set(rgba or rgb)
        if ($opacity) {
            if (abs($opacity) > 1) {
                $opacity = 1.0;
            }

            $output = 'rgba(' . implode(',', $rgb) . ',' . $opacity . ')';
        } else {
            $output = 'rgb(' . implode(',', $rgb) . ')';
        }

        //Return rgb(a) color string
        return $output;
    }

    /**
     * Fired when the `TemplateManager::display` hook is called.
     *
     * @param string $hookname
     * @param array $args [$templateMgr, $template, $sendContentType, $charset, $output]
     */
    public function loadTemplateData($hookName, $args)
    {
        $application = Application::get();
        // Retrieve the TemplateManager
        $templateMgr = $args[0];
        $ojt         = $this->openJournalThemeInit();
        if (!$ojt->isFromLocal() && $this->isActive() && !$ojt->isMainIndexPage()) {
            $ojt->main();
            if (!$ojt->isNotDomain()) {
                if ($ojt->getStatusValidated() == 2) {
                    $templateMgr->assign('prevent', $ojt->prevent());
                }
                $templateMgr->assign('alertValidated', $ojt->getHtml());
            }
        }

        //rtl mode
        $locale    = AppLocale::getLocale();
        $isRTLMode = (AppLocale::getLocaleDirection($locale) === 'rtl') ? true : false;
        $templateMgr->assign('rtlMode', $isRTLMode);

        $baseUrl = Config::getVar('general', 'base_url');

        // Attach a custom piece of data to the TemplateManager
        $myCustomData = 'This is my custom data. It could be any PHP variable.';
        $templateMgr->assign('pluginDir', $this->getPluginPath());
        $templateMgr->assign('pluginImageDir', $baseUrl . '/' . $this->getPluginPath() . '/images/');
        $templateMgr->assign('enableStatistic', $this->getOption('statistics'));
        $templateMgr->assign('articleTrackingDate', $this->getOption('article_tracking_date'));
        $templateMgr->assign('enableGalleyLink', $this->getOption('galley_link_on_index'));

        $logoStyle = $this->getOption('logoStyleBig');
        $templateMgr->assign('logoStyleBig', $logoStyle);

        $templateMgr->assign('dimensionBadge', $this->getOption('dimension_badge'));
        $templateMgr->assign('showDoiOnIndex', $this->getOption('doi_on_index'));
        $templateMgr->assign('journalVersion', $this->getVersionNumber());

        $enableCrossMarkButton = $this->getOption('crossmark_button');
        $templateMgr->assign('enableCrossMarkButton', $enableCrossMarkButton);

        $contextSettings      = $templateMgr->get_template_vars('contextSettings');
        $publisherInstitution = isset($contextSettings['publisherInstitution']) ? $contextSettings['publisherInstitution'] : false;
        $templateMgr->assign('publisherName', $publisherInstitution);


        // Retrieve the TemplateManager
        $submission   = $templateMgr->get_template_vars('article');
        if ($submission) {
            $submissionId = $submission->getData('id');

            $dateAccepted = '';

            $editDecisionDao = DAORegistry::getDAO('EditDecisionDAO');
            $decisions       = $editDecisionDao->getEditorDecisions($submissionId);

            foreach ($decisions as $decision) {
                if ($decision['stageId'] == '3' && $decision['decision'] == '1') {
                    $dateAccepted = (isset($decision['dateDecided'])) ? $decision['dateDecided'] : false;
                }
            }

            if ($dateAccepted) {
                $templateMgr->assign('acceptanceDate', $dateAccepted);
            }
        }

        //based on google scholar guidlines to add ol tag
        $article = $templateMgr->get_template_vars('article');
        if ($article) {
            $references = $article->getCitations();
            $references = explode(PHP_EOL, $references);
            $templateMgr->assign('references', $references);
        }

        $publication_data = $templateMgr->get_template_vars('currentPublication');

        $all = $templateMgr->get_template_vars('publishedSubmissions');

        //next plugin dimension badge
        $article_doi = isset($publication_data->_data['pub-id::doi']) ? $publication_data->_data['pub-id::doi'] : false;
        $templateMgr->assign('article_doi', $article_doi);

        $request = $application->getRequest();
        $context = $request->getContext();

        $context_data_id = ($context) ? $context->_data['id'] : false;

        $templateMgr->assign('context_id', $context_data_id);

        //maintenance mode
        $maintenance_mode = $this->getOption('maintenance_mode');
        $templateMgr->assign('isMaintenance', $maintenance_mode);

        //disable cache on openjournaltheme server level
        if ($maintenance_mode) {
            //disable cache on server level - nginx fastcgi cache
            setcookie('OJTNOCACHE', 1, time() + 60, '/');
            unset($_COOKIE['MAKECACHE']);
        }
        if (!$maintenance_mode) {
            setcookie('MAKECACHE', 1, time() + 60, '/');
            unset($_COOKIE['OJTNOCACHE']);
        }
    }

    private function getVersionNumber()
    {
        $versionDao = DAORegistry::getDAO('VersionDAO');
        $version    = $versionDao->getCurrentVersion();
        $data       = $version->_data;
        return $data['major'] . $data['minor'];
    }

    public function vd($data)
    {
        highlight_string("<?php\n\$data =\n" . var_export($data, true) . ";\n?>");
    }

    public function getPluginVersion()
    {
        import('lib.pkp.classes.site.VersionCheck');
        $version = VersionCheck::parseVersionXML($this->getPluginVersionFile());

        return $version['release'];
    }

    public function getPluginVersionFile()
    {
        return $this->getPluginPath() . '/version.xml';
    }
}
