<?php

import('lib.pkp.classes.plugins.GenericPlugin');

use JATSParser\HTML\Document as HTMLDocument;
use JATSParser\Body\Document;

class OjtJatsParser extends GenericPlugin
{
  public function register($category, $path, $mainContextId = null)
  {
    if (parent::register($category, $path, $mainContextId)) {
      if ($this->getEnabled()) {
        if (!class_exists('\JATSParser\Body\Document', true)) {
          require_once  __DIR__ . '/JATSParser/vendor/autoload.php';
        }
        HookRegistry::register('ArticleHandler::view::galley', array($this, 'articleGalley'), HOOK_SEQUENCE_LATE);
        HookRegistry::register('Templates::Article::Main', array($this, 'displayFullText'));
        HookRegistry::register('LoadHandler', array($this, 'loadFullTextAssocHandler'));
      }
      return true;
    }
    return false;
  }

  /**
   * Get the display name of this plugin.
   * @return String
   */
  public function getDisplayName()
  {
    return 'OJT Jats Parser';
  }

  /**
   * Get a description of the plugin.
   */
  public function getDescription()
  {
    return "The plugin transforms JATS XML into HTML and PDF for viewing on the journal's front-end";
  }

  public function displayFullText(string $hookName, array $args)
  {
    $templateMgr = &$args[1];
    $output = &$args[2];

    $publication = $templateMgr->getTemplateVars('publication');

    $embeddedXml = null;

    // find PDF or XML in the current locale
    foreach ($publication->getData('galleys') as $localizedGalley) {
      if ($localizedGalley->getFileType() === "application/xml" || $localizedGalley->getFileType() === "text/xml") {
        $embeddedXml = $localizedGalley;
      }
    }

    if (!$embeddedXml) return false;

    $submissionFile = $embeddedXml->getFile();

    $jatsDocument = new Document(Config::getVar('files', 'files_dir') . '/' .  $submissionFile->getData('path'));
    $htmlDocument = $this->htmlCreation($templateMgr, $jatsDocument, $embeddedXml);

    $templateMgr->assign('htmlDocument', $htmlDocument->getHmtlForGalley());
    $output .= $templateMgr->fetch($this->getTemplateResource('jats.tpl'));
  }

  /**
   * @param $jatsDocument Document
   * @param $templateMgr TemplateManager
   * @return HTMLDocument HTMLDocument
   */
  protected function htmlCreation($templateMgr, $jatsDocument, $embeddedXml): HTMLDocument
  {
    // HTML DOM
    $htmlDocument = new HTMLDocument($jatsDocument);

    $submissionFile     = $embeddedXml->getFile();
    $submissionId       = $submissionFile->getData('submissionId');

    $embeddableFiles = Services::get('submissionFile')->getMany([
      'assocTypes' => [ASSOC_TYPE_SUBMISSION_FILE],
      'assocIds' => [$submissionFile->getId()],
      'submissionIds' => [$submissionFile->getData('submissionId')],
      'fileStages' => [SUBMISSION_FILE_DEPENDENT],
      'includeDependentFiles' => true,
    ]);

    // Add the link to images
    $imageUrlArray = array();

    foreach ($embeddableFiles as $embeddableFile) {
      if ($embeddableFile->getData('mimetype') == 'image/png' || $embeddableFile->getData('mimetype') == 'image/jpeg') {

        $fileUrl = $this->getRequest()->url(null, 'article', 'downloadFullTextAssoc', array($submissionId, $embeddableFile->getData('assocId'), $embeddableFile->getData('id')));

        $imageUrlArray[$embeddableFile->getLocalizedData('name')] = $fileUrl;
      }
    }



    // Replace link with actual path
    $xpath = new \DOMXPath($htmlDocument);
    $imageLinks = $xpath->evaluate("//img");
    foreach ($imageLinks as $imageLink) {
      if ($imageLink->hasAttribute("src")) {
        array_key_exists($imageLink->getAttribute("src"), $imageUrlArray);
        $imageLink->setAttribute("src", $imageUrlArray[$imageLink->getAttribute("src")]);
      }
    }


    // Localization of reference list title
    $referenceTitles = $xpath->evaluate("//h2[@id='reference-title']");
    $translateReference = $templateMgr->smartyTranslate(array('key' => 'submission.citations'), $templateMgr);
    if ($referenceTitles->length > 0) {
      foreach ($referenceTitles as $referenceTitle) {
        $referenceTitle->nodeValue = $translateReference;
      }
    }

    // Special treatment for table head
    $tableHeadRows = $xpath->evaluate("//thead/tr");

    foreach ($tableHeadRows as $tableHeadRow) {
      $tableHeadRow->setAttribute("align", "center");
      $tableHeadRow->setAttribute("style", "background-color:#f2e6ff;");
    }


    return $htmlDocument;
  }


  /**
   * @param $hookName string
   * @param $args array
   * @brief Handle associated files of the full-text, only images are supported
   */
  function loadFullTextAssocHandler($hookName, $args)
  {
    $page = $args[0];
    $op = $args[1];

    if ($page == 'article' && $op == 'downloadFullTextAssoc') {
      define('HANDLER_CLASS', 'FullTextArticleHandler');
      $args[2] = $this->getPluginPath() . DIRECTORY_SEPARATOR . 'FullTextArticleHandler.inc.php';
    }
  }

  function articleGalley($hookName, $args)
  {
    $request = &$args[0];
    $issue = &$args[1];
    $galley = &$args[2];
    $submission = &$args[3];

    $templateMgr = TemplateManager::getManager($request);


    if (!$galley || !in_array($galley->getFileType(), array('application/xml', 'text/xml'))) {
      return false;
    }
    $publication = $templateMgr->getTemplateVars('publication');

    $submissionFile = $galley->getFile();

    $jatsDocument = new Document(Config::getVar('files', 'files_dir') . '/' .  $submissionFile->getData('path'));
    $htmlDocument = $this->htmlCreation($templateMgr, $jatsDocument, $galley);

    $templateMgr->assign('htmlDocument', $htmlDocument->getHmtlForGalley());

    $templateMgr->display($this->getTemplateResource('articleGalley.tpl'));
    return true;
  }
}
